<?php

namespace App\Http\Controllers;

use App\Models\Booking;
use App\Models\ScheduleSlot;
use Illuminate\Http\Request;

class BookingController extends Controller
{
    /* =========================
     * INDEX
     * ========================= */
    public function index()
    {
        $authUser = auth()->user();

        $query = Booking::with([
            'slot.schedule.company',
            'slot.schedule.user',
            'user'
        ])->latest();

        if ($authUser->hasRole('Super Admin')) {
            // See all bookings
        }
        elseif ($authUser->can('bookings.view.all')) {
            $query->whereHas('slot.schedule', function ($q) use ($authUser) {
                $q->where('company_id', $authUser->profile->company_id);
            });
        }
        elseif ($authUser->can('bookings.view.own')) {
            $query->where('user_id', $authUser->id);
        }
        else {
            abort(403);
        }

        $bookings = $query->paginate(15);

        return view('bookings.index', compact('bookings'));
    }

    /* =========================
     * CREATE
     * ========================= */
    public function create(Request $request)
    {
        $slot = null;
        $schedule = null;
        $scheduledUser = null;

        if ($request->filled('slot_id')) {
            $slot = ScheduleSlot::with('schedule.user', 'schedule.company')
                ->findOrFail($request->slot_id);

            $schedule = $slot->schedule;
            $scheduledUser = $schedule->user;
        }

        return view('bookings.create', compact('slot', 'schedule', 'scheduledUser'));
    }

    /* =========================
     * STORE
     * ========================= */
    public function store(Request $request)
    {
        $authUser = auth()->user();

        $data = $request->validate([
            'slot_id' => 'required|exists:schedule_slots,id',
            'customer_name' => 'required|string|max:255',
            'customer_phone' => 'required|string|max:20',
            'customer_reference' => 'nullable|string|max:255',
            'data' => 'nullable|string',
        ]);

        $slot = ScheduleSlot::with('schedule')->findOrFail($data['slot_id']);

        /* 🔒 Company-level access check */
        if (
            !$authUser->hasRole('Super Admin') &&
            $slot->schedule->company_id !== $authUser->profile->company_id
        ) {
            abort(403);
        }

        // Prevent double booking
        if (Booking::where('slot_id', $data['slot_id'])->exists()) {
            return back()->withInput()
                ->with('error', 'This slot has already been booked.');
        }

        // Convert JSON data
        if (!empty($data['data'])) {
            $data['data'] = json_decode($data['data'], true) ?: null;
        }

        $data['user_id'] = $authUser->id;
        $data['stage'] = 'new';
        $data['status'] = 'pending';

        Booking::create($data);

        return redirect()
            ->route('admin.bookings.index')
            ->with('success', 'Booking created successfully.');
    }

    /* =========================
     * SHOW
     * ========================= */
    public function show(Booking $booking)
    {
        $booking->load([
            'slot.schedule.company',
            'slot.schedule.user',
            'user'
        ]);

        $authUser = auth()->user();

        if (
            !$authUser->hasRole('Super Admin') &&
            $booking->slot->schedule->company_id !== $authUser->profile->company_id
        ) {
            abort(403);
        }

        return view('bookings.show', compact('booking'));
    }

    /* =========================
     * EDIT
     * ========================= */
    public function edit(Booking $booking)
    {
        $booking->load('slot.schedule');

        $authUser = auth()->user();

        if (
            !$authUser->hasRole('Super Admin') &&
            $booking->slot->schedule->company_id !== $authUser->profile->company_id
        ) {
            abort(403);
        }

        $slot = $booking->slot;

        return view('bookings.create', compact('booking', 'slot'));
    }

    /* =========================
     * UPDATE
     * ========================= */
    public function update(Request $request, Booking $booking)
    {
        $booking->load('slot.schedule');

        $authUser = auth()->user();

        if (
            !$authUser->hasRole('Super Admin') &&
            $booking->slot->schedule->company_id !== $authUser->profile->company_id
        ) {
            abort(403);
        }

        $data = $request->validate([
            'customer_name' => 'required|string|max:255',
            'customer_phone' => 'required|string|max:20',
            'customer_reference' => 'nullable|string|max:255',
            'stage' => 'required|string',
            'status' => 'required|string',
            'data' => 'nullable|string',
        ]);

        if (!empty($data['data'])) {
            $data['data'] = json_decode($data['data'], true) ?: null;
        }

        $booking->update($data);

        return redirect()
            ->route('admin.bookings.index')
            ->with('success', 'Booking updated successfully.');
    }

    /* =========================
     * DELETE
     * ========================= */
    public function destroy(Booking $booking)
    {
        $booking->load('slot.schedule');

        $authUser = auth()->user();

        if (
            !$authUser->hasRole('Super Admin') &&
            $booking->slot->schedule->company_id !== $authUser->profile->company_id
        ) {
            abort(403);
        }

        $booking->delete();

        return redirect()
            ->route('admin.bookings.index')
            ->with('success', 'Booking cancelled successfully.');
    }
}
