<?php

namespace App\Http\Controllers;

use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Str;
use Illuminate\View\View;
use Illuminate\Support\Facades\Storage;

class CompanyController extends Controller
{
    public function index(): View
    {
        $companies = Company::orderBy('order')->paginate(20);
        return view('companies.index', compact('companies'));
    }

    public function create(): View
    {
        $parents = Company::all(); // For parent dropdown
        return view('companies.create', compact('parents'));
    }

    public function store(Request $request): RedirectResponse
    {
        $data = $this->validatedData($request);

        $data['slug'] = $data['slug'] ?: Str::slug($data['name']);

        if ($request->hasFile('logo')) {
            $data['logo'] = $request->file('logo')->store('companies/logos', 'public');
        }

        if ($request->hasFile('banner')) {
            $data['banner'] = $request->file('banner')->store('companies/banners', 'public');
        }

        Company::create($data);

        return redirect()->route('admin.companies.index')->with('success', 'Company created successfully.');
    }

    public function edit(Company $company): View
    {
        $parents = Company::where('id', '!=', $company->id)->get(); // Exclude self
        return view('companies.edit', compact('company', 'parents'));
    }

    public function update(Request $request, Company $company): RedirectResponse
    {
        $data = $this->validatedData($request, $company->id);

        $data['slug'] = $data['slug'] ?: Str::slug($data['name']);

        if ($request->hasFile('logo')) {
            if ($company->logo) Storage::disk('public')->delete($company->logo);
            $data['logo'] = $request->file('logo')->store('companies/logos', 'public');
        }

        if ($request->hasFile('banner')) {
            if ($company->banner) Storage::disk('public')->delete($company->banner);
            $data['banner'] = $request->file('banner')->store('companies/banners', 'public');
        }

        $company->update($data);

        return redirect()->route('admin.companies.index')->with('success', 'Company updated successfully.');
    }

    public function destroy(Company $company): RedirectResponse
    {
        if ($company->logo) Storage::disk('public')->delete($company->logo);
        if ($company->banner) Storage::disk('public')->delete($company->banner);

        $company->delete();

        return back()->with('success', 'Company deleted successfully.');
    }

    protected function validatedData(Request $request, int $id = null): array
    {
        return $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|unique:companies,slug,' . $id,
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:50',
            'website' => 'nullable|url|max:255',
            'parent_id' => 'nullable|exists:companies,id',
            'logo' => 'nullable|file|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'banner' => 'nullable|file|image|mimes:jpeg,png,jpg,gif,webp|max:4096',
            'short_description' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'status' => 'boolean',
            'order' => 'nullable|integer',
        ]);
    }
}
