<?php

namespace App\Http\Controllers;

use App\Models\Lead;
use App\Models\LeadStage;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;

class LeadController extends Controller
{
    /** Display a list of leads */
    public function index()
    {
        $user = auth()->user();

        $query = Lead::with(['stage', 'user', 'company'])->latest();

        // Permission-based filtering
        if ($user->can('leads.view.all')) {
            // Super Admin: see all
        } 
        elseif ($user->can('leads.view.company')) {
            $companyId = optional($user->profile)->company_id;
            if (!$companyId) abort(403, 'User has no company.');
            $query->where('company_id', $companyId);
        } 
        elseif ($user->can('leads.view.own')) {
            $query->where('user_id', $user->id);
        } 
        else {
            abort(403, 'Unauthorized');
        }

        $leads  = $query->paginate(15);
        $stages = LeadStage::orderBy('order')->get();

        // Only users with change_user permission can see user list for bulk assign
        $users = $user->can('leads.change_user')
            ? User::orderBy('name')->get()
            : collect();

        return view('leads.index', compact('leads', 'stages', 'users'));
    }

    /** Show create lead form */
    public function create()
    {
        if (Gate::denies('leads.create')) {
            abort(403);
        }

        $user = auth()->user();

        $stages = LeadStage::orderBy('order')->get();

        $users = $user->can('leads.change_user')
            ? User::orderBy('name')->get()
            : collect();

        return view('leads.create', compact('stages', 'users'));
    }

    /** Store new lead */
    public function store(Request $request)
    {
        if (Gate::denies('leads.create')) {
            abort(403);
        }

        $data = $request->validate([
            'name'           => 'required|string|max:255',
            'phone'          => 'required|string|max:20',
            'email'          => 'nullable|email',
            'note'           => 'nullable|string',
            'reference'      => 'nullable|string|max:255',
            'portal'         => 'nullable|string|max:255',
            'portal_lead_id' => 'nullable|string|max:255',
            'stage_id'       => 'nullable|exists:lead_stages,id',
            'company_id'     => 'nullable|exists:companies,id',
            'data'           => 'nullable|string',
        ]);

        if (!empty($data['data'])) {
            $data['data'] = json_decode($data['data'], true);
        }

        $data['user_id'] = auth()->id();

        // Auto-assign company if not provided
        if (!isset($data['company_id']) && auth()->user()->profile) {
            $data['company_id'] = auth()->user()->profile->company_id;
        }

        Lead::create($data);

        return redirect()->route('admin.leads.index')
            ->with('success', 'Lead created successfully');
    }

    /** Show lead details */
    public function show(Lead $lead)
    {
        $this->authorizeLead($lead);

        $lead->load(['stage', 'user', 'company']);
        $stages = LeadStage::orderBy('order')->get();

        $users = auth()->user()->can('leads.change_user')
            ? User::orderBy('name')->get()
            : collect();

        return view('leads.show', compact('lead', 'users', 'stages'));
    }

    /** Show edit lead form */
    public function edit(Lead $lead)
    {
        if (Gate::denies('leads.edit')) {
            abort(403);
        }

        $this->authorizeLead($lead);

        $stages = LeadStage::orderBy('order')->get();

        $users = auth()->user()->can('leads.change_user')
            ? User::orderBy('name')->get()
            : collect();

        return view('leads.edit', compact('lead', 'stages', 'users'));
    }

    /** Update lead */
    public function update(Request $request, Lead $lead)
    {
        if (Gate::denies('leads.edit')) {
            abort(403);
        }

        $this->authorizeLead($lead);

        $data = $request->validate([
            'name'       => 'required|string|max:255',
            'phone'      => 'required|string|max:20',
            'email'      => 'nullable|email',
            'note'       => 'nullable|string',
            'reference'  => 'nullable|string|max:255',
            'portal'     => 'nullable|string|max:255',
            'stage_id'   => 'nullable|exists:lead_stages,id',
            'company_id' => 'nullable|exists:companies,id',
            'data'       => 'nullable|string',
            'user_id'    => 'nullable|exists:users,id',
        ]);

        if (!empty($data['data'])) {
            $data['data'] = json_decode($data['data'], true);
        }

        $lead->update($data);

        return redirect()->route('admin.leads.index')
            ->with('success', 'Lead updated successfully');
    }

    /** Bulk change stage */
    public function bulkStageUpdate(Request $request)
    {
        if (Gate::denies('leads.change_stage')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $request->validate([
            'lead_ids'     => 'required|array',
            'new_stage_id' => 'required|exists:lead_stages,id',
        ]);

        Lead::whereIn('id', $request->lead_ids)
            ->update(['stage_id' => $request->new_stage_id]);

        return response()->json(['success' => true]);
    }

    /** Bulk change user */
    public function bulkUserUpdate(Request $request)
    {
        if (Gate::denies('leads.change_user')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $request->validate([
            'lead_ids'   => 'required|array',
            'new_user_id'=> 'required|exists:users,id',
        ]);

        Lead::whereIn('id', $request->lead_ids)
            ->update(['user_id' => $request->new_user_id]);

        return response()->json(['success' => true]);
    }

    /** Bulk delete leads */
    public function bulkDelete(Request $request)
    {
        if (Gate::denies('leads.delete')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $request->validate([
            'lead_ids' => 'required|array',
        ]);

        Lead::whereIn('id', $request->lead_ids)->delete();

        return response()->json(['success' => true]);
    }

    /** Delete a lead */
    public function destroy(Lead $lead)
    {
        if (Gate::denies('leads.delete')) {
            abort(403);
        }

        $this->authorizeLead($lead);

        $lead->delete();

        return redirect()->route('admin.leads.index')
            ->with('success', 'Lead deleted successfully');
    }

    /** Shared authorization logic */
    private function authorizeLead(Lead $lead)
    {
        $user = auth()->user();

        if ($user->hasRole('Super Admin')) return true;

        if ($user->can('leads.view.company') &&
            $lead->company_id === optional($user->profile)->company_id) {
            return true;
        }

        if ($user->can('leads.view.own') &&
            $lead->user_id === $user->id) {
            return true;
        }

        abort(403, 'Unauthorized');
    }
}
