<?php

namespace App\Http\Controllers;

use App\Models\LeadStage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;

class LeadStageController extends Controller
{
    public function index()
    {

        $user = auth()->user();

        $query = LeadStage::with('parent')->orderBy('order');

        // 🔐 Permission-based filtering
        if ($user->can('leadstages.view.all')) {
            // Super Admin → all stages
        }
        elseif ($user->can('leadstages.view.company')) {
            $companyId = optional($user->profile)->company_id;
            if (!$companyId) abort(403);
            $query->where('company_id', $companyId);
        }
        else {
            abort(403);
        }

        $stages = $query->paginate(15);

        return view('lead_stages.index', compact('stages'));
    }

    // public function kanban()
    // {
    //     $user = auth()->user();

    //     $query = LeadStage::with('parent')
    //         ->whereNull('parent_id') // Columns (main stages)
    //         ->orderBy('order');

    //     if ($user->can('leadstages.view.all')) {
    //         // all
    //     } elseif ($user->can('leadstages.view.company')) {
    //         $companyId = optional($user->profile)->company_id;
    //         if (!$companyId) abort(403);
    //         $query->where('company_id', $companyId);
    //     } else {
    //         abort(403);
    //     }

    //     $stages = $query->with([
    //         'children' => function ($q) {
    //             $q->orderBy('order');
    //         }
    //     ])->get();

    //     return view('lead_stages.kanban', compact('stages'));
    // }


    public function create()
    {
        if (Gate::denies('leadstages.create')) {
            abort(403);
        }

        $user = auth()->user();
        $companyId = optional($user->profile)->company_id;

        $parents = LeadStage::where('status', 1)
            ->when(!$user->can('leadstages.view.all'), function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            })
            ->get();

        return view('lead_stages.create', compact('parents'));
    }

    public function store(Request $request)
    {
        if (Gate::denies('leadstages.create')) {
            abort(403);
        }

        $user = auth()->user();

        $data = $request->validate([
            'name'       => 'required|string|max:255',
            'parent_id'  => 'nullable|exists:lead_stages,id',
            'company_id' => 'nullable|exists:companies,id',
            'order'      => 'nullable|integer',
            'status'     => 'required|boolean',
            'data'       => 'nullable|string',
        ]);

        if (!empty($data['data'])) {
            $data['data'] = json_decode($data['data'], true);
        }

        // 🔒 Force company for non-super-admin
        if (!$user->can('leadstages.view.all')) {
            $data['company_id'] = optional($user->profile)->company_id;
        }

        LeadStage::create($data);

        return redirect()
            ->route('admin.lead-stages.index')
            ->with('success', 'Lead stage created successfully.');
    }

    public function edit(LeadStage $leadStage)
    {
        if (Gate::denies('leadstages.edit')) {
            abort(403);
        }

        $this->authorizeStage($leadStage);

        $user = auth()->user();
        $companyId = optional($user->profile)->company_id;

        $parents = LeadStage::where('id', '!=', $leadStage->id)
            ->when(!$user->can('leadstages.view.all'), function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            })
            ->get();

        return view('lead_stages.edit', [
            'stage'   => $leadStage,
            'parents' => $parents,
        ]);
    }

    public function update(Request $request, LeadStage $leadStage)
    {
        if (Gate::denies('leadstages.edit')) {
            abort(403);
        }

        $this->authorizeStage($leadStage);

        $data = $request->validate([
            'name'       => 'required|string|max:255',
            'parent_id'  => 'nullable|exists:lead_stages,id',
            'company_id' => 'nullable|exists:companies,id',
            'order'      => 'nullable|integer',
            'status'     => 'required|boolean',
            'data'       => 'nullable|string',
        ]);

        if (!empty($data['data'])) {
            $data['data'] = json_decode($data['data'], true);
        }

        // 🔒 Prevent company change unless super admin
        if (!auth()->user()->can('leadstages.view.all')) {
            unset($data['company_id']);
        }

        $leadStage->update($data);

        return redirect()
            ->route('admin.lead-stages.index')
            ->with('success', 'Lead stage updated successfully.');
    }

    public function destroy(LeadStage $leadStage)
    {
        if (Gate::denies('leadstages.delete')) {
            abort(403);
        }

        $this->authorizeStage($leadStage);

        $leadStage->delete();

        return redirect()
            ->route('admin.lead-stages.index')
            ->with('success', 'Lead stage deleted successfully.');
    }

    /**
     * 🔐 Shared authorization logic
     */
    private function authorizeStage(LeadStage $stage)
    {
        $user = auth()->user();

        if ($user->can('leadstages.view.all')) {
            return true;
        }

        if ($user->can('leadstages.view.company') &&
            $stage->company_id === optional($user->profile)->company_id) {
            return true;
        }

        abort(403);
    }
}
