<?php

namespace App\Http\Controllers;

use App\Models\Location;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class LocationController extends Controller
{
    public function index()
    {
        $locations = Location::orderBy('order')->paginate(15);
        return view('locations.index', compact('locations'));
    }

    public function create()
    {
        $parents = Location::all();
        return view('locations.create', compact('parents'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'title' => 'nullable|string|max:255',
            'slug' => 'nullable|string|max:255|unique:locations,slug',
            'parent_id' => 'nullable|exists:locations,id',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'image' => 'nullable|image|max:2048',
            'banner' => 'nullable|image|max:2048',
            'icon' => 'nullable|image|max:2048',
            'short_description' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'meta' => 'nullable|array',
            'data' => 'nullable|array',
            'status' => 'boolean',
            'order' => 'integer',
        ]);

        // Auto-generate slug if not provided
        if (empty($data['slug'])) {
            $data['slug'] = $this->generateUniqueSlug($data['name']);
        }

        // Handle uploads
        foreach (['image','banner','icon'] as $file) {
            if ($request->hasFile($file)) {
                $data[$file] = $request->file($file)->store('locations');
            }
        }

        Location::create($data);

        return redirect()->route('admin.locations.index')
                         ->with('success', 'Location created successfully.');
    }

    public function edit(Location $location)
    {
        $parents = Location::where('id', '!=', $location->id)->get();
        return view('locations.edit', compact('location', 'parents'));
    }

    public function update(Request $request, Location $location)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'title' => 'nullable|string|max:255',
            'slug' => 'nullable|string|max:255|unique:locations,slug,' . $location->id,
            'parent_id' => 'nullable|exists:locations,id',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'image' => 'nullable|image|max:2048',
            'banner' => 'nullable|image|max:2048',
            'icon' => 'nullable|image|max:2048',
            'short_description' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'meta' => 'nullable|array',
            'data' => 'nullable|array',
            'status' => 'boolean',
            'order' => 'integer',
        ]);

        // Auto-generate slug if not provided
        if (empty($data['slug'])) {
            $data['slug'] = $this->generateUniqueSlug($data['name'], $location->id);
        }

        // Handle uploads
        foreach (['image','banner','icon'] as $file) {
            if ($request->hasFile($file)) {
                $data[$file] = $request->file($file)->store('locations');
            }
        }

        $location->update($data);

        return redirect()->route('admin.locations.index')
                         ->with('success', 'Location updated successfully.');
    }

    public function destroy(Location $location)
    {
        $location->delete();
        return redirect()->route('admin.locations.index')
                         ->with('success', 'Location deleted successfully.');
    }

    public function show(Location $location)
    {
        return view('locations.show', compact('location'));
    }

    /**
     * Generate a unique slug for a location
     *
     * @param string $name
     * @param int|null $ignoreId
     * @return string
     */
    protected function generateUniqueSlug(string $name, int $ignoreId = null): string
    {
        $slug = Str::slug($name);
        $original = $slug;
        $i = 1;

        while (Location::where('slug', $slug)
            ->when($ignoreId, fn($q) => $q->where('id', '!=', $ignoreId))
            ->exists()) {
            $slug = $original . '-' . $i++;
        }

        return $slug;
    }
}
