<?php

namespace App\Http\Controllers;

use App\Models\Pipeline;
use App\Models\LeadStage;
use Illuminate\Http\Request;

class PipelineController extends Controller
{
    /**
     * Display pipelines available for the company
     */
    public function index()
    {
        $user = auth()->user();

        $pipelines = Pipeline::with('stage')
            ->where('company_id', $user->company_id)
            ->orderBy('id', 'desc')
            ->get();

        return view('pipelines.index', compact('pipelines'));
    }

    /**
     * Show create form
     */
    public function create()
    {
        $user = auth()->user();

        $stages = LeadStage::where('company_id', $user->company_id)
            ->where('status', true)
            ->orderBy('order')
            ->get();

        return view('pipelines.create', compact('stages'));
    }

    /**
     * Store new pipeline
     */
    public function store(Request $request)
    {
        $request->validate([
            'name'     => 'required|string|max:255',
            'stage_id' => 'nullable|exists:lead_stages,id',
            'status'   => 'required|boolean',
        ]);

        Pipeline::create([
            'name'       => $request->name,
            'status'     => $request->status,
            'stage_id'   => $request->stage_id,
            'company_id' => auth()->user()->company_id,
            'user_id'    => auth()->id(),
        ]);

        return redirect()
            ->route('pipelines.index')
            ->with('success', 'Pipeline created successfully');
    }

    /**
     * Show edit form
     */
    public function edit(Pipeline $pipeline)
    {
        $this->authorizeCompany($pipeline);

        $stages = LeadStage::where('company_id', auth()->user()->company_id)
            ->where('status', true)
            ->orderBy('order')
            ->get();

        return view('pipelines.edit', compact('pipeline', 'stages'));
    }

    /**
     * Update pipeline
     */
    public function update(Request $request, Pipeline $pipeline)
    {
        $this->authorizeCompany($pipeline);

        $request->validate([
            'name'     => 'required|string|max:255',
            'stage_id' => 'nullable|exists:lead_stages,id',
            'status'   => 'required|boolean',
        ]);

        $pipeline->update($request->only([
            'name',
            'status',
            'stage_id',
        ]));

        return redirect()
            ->route('pipelines.index')
            ->with('success', 'Pipeline updated successfully');
    }

    /**
     * Delete pipeline
     */
    public function destroy(Pipeline $pipeline)
    {
        $this->authorizeCompany($pipeline);

        $pipeline->delete();

        return back()->with('success', 'Pipeline deleted');
    }

    /**
     * Company scope security
     */
    private function authorizeCompany(Pipeline $pipeline)
    {
        if ($pipeline->company_id !== auth()->user()->company_id) {
            abort(403);
        }
    }
}
