<?php

namespace App\Http\Controllers;

use App\Models\PropertyDeveloper;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PropertyDeveloperController extends Controller
{
    public function index()
    {
        $developers = PropertyDeveloper::orderBy('order')->paginate(15);
        return view('property_developers.index', compact('developers'));
    }

    public function create()
    {
        return view('property_developers.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:property_developers,slug',
            'title' => 'nullable|string|max:255',
            'short_description' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'meta' => 'nullable|array',
            'data' => 'nullable|array',
            'logo' => 'nullable|image|max:2048',
            'banner' => 'nullable|image|max:2048',
            'icon' => 'nullable|image|max:2048',
            'status' => 'boolean',
            'order' => 'integer',
        ]);

        // Auto-generate slug if not provided
        $data['slug'] = $data['slug'] ?? Str::slug($data['name']);

        // Handle file uploads
        foreach (['logo','banner','icon'] as $file) {
            if ($request->hasFile($file)) {
                $data[$file] = $request->file($file)->store('property_developers');
            }
        }

        $developer = PropertyDeveloper::create($data);

        return redirect()->route('admin.property_developers.index')
                         ->with('success', 'Property Developer created successfully.');
    }

    public function edit(PropertyDeveloper $propertyDeveloper)
    {
        return view('property_developers.edit', compact('propertyDeveloper'));
    }

    public function update(Request $request, PropertyDeveloper $propertyDeveloper)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:property_developers,slug,' . $propertyDeveloper->id,
            'title' => 'nullable|string|max:255',
            'short_description' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'meta' => 'nullable|array',
            'data' => 'nullable|array',
            'logo' => 'nullable|image|max:2048',
            'banner' => 'nullable|image|max:2048',
            'icon' => 'nullable|image|max:2048',
            'status' => 'boolean',
            'order' => 'integer',
        ]);

        // Auto-generate slug if not provided
        $data['slug'] = $data['slug'] ?? Str::slug($data['name']);

        // Handle file uploads
        foreach (['logo','banner','icon'] as $file) {
            if ($request->hasFile($file)) {
                $data[$file] = $request->file($file)->store('property_developers');
            }
        }

        $propertyDeveloper->update($data);

        return redirect()->route('admin.property_developers.index')
                         ->with('success', 'Property Developer updated successfully.');
    }

    public function destroy(PropertyDeveloper $propertyDeveloper)
    {
        $propertyDeveloper->delete();
        return redirect()->route('admin.property_developers.index')
                         ->with('success', 'Property Developer deleted successfully.');
    }

    public function show(PropertyDeveloper $propertyDeveloper)
    {
        return view('property_developers.show', compact('propertyDeveloper'));
    }
}
