<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;

class RoleController extends Controller
{
    /**
     * Apply middleware to protect routes
     */
    // public function __construct()
    // {
    //     // Only users with 'manage roles' permission can access these routes
    //     $this->middleware('permission:manage roles');
    // }

    /**
     * Display a listing of roles
     */
    public function index()
    {
        // Load roles with their permissions
        $roles = Role::with('permissions')->latest()->get();

        return view('roles.index', compact('roles'));
    }

    /**
     * Show the form for creating a new role
     */
    public function create()
    {
        // Get all available permissions
        $permissions = Permission::all();

        return view('roles.create', compact('permissions'));
    }

    /**
     * Store a newly created role in storage
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|unique:roles,name',
            'permissions' => 'array'
        ]);

        // Create the role
        $role = Role::create(['name' => $request->name]);

        // Assign permissions if provided
        if ($request->has('permissions')) {
            $role->syncPermissions($request->permissions);
        }

        return redirect()->route('admin.roles.index')
                         ->with('success', 'Role created successfully.');
    }

    /**
     * Show the form for editing the specified role
     */
    public function edit(Role $role)
    {
        $permissions = Permission::all();
        $rolePermissions = $role->permissions->pluck('name')->toArray();

        return view('roles.edit', compact('role', 'permissions', 'rolePermissions'));
    }

    /**
     * Update the specified role in storage
     */
    public function update(Request $request, Role $role)
    {
        $request->validate([
            'name' => 'required|string|unique:roles,name,' . $role->id,
            'permissions' => 'array'
        ]);

        // Update role name
        $role->update(['name' => $request->name]);

        // Sync permissions
        $role->syncPermissions($request->permissions ?? []);

        return redirect()->route('admin.roles.index')
                         ->with('success', 'Role updated successfully.');
    }

    public function show($id)
{
    abort(404); // or return a view if you want to display role details
}

    /**
     * Remove the specified role from storage
     */
    public function destroy(Role $role)
    {
        $role->delete();

        return redirect()->route('admin.roles.index')
                         ->with('success', 'Role deleted successfully.');
    }
}
