<?php

namespace App\Http\Controllers;

use App\Models\Schedule;
use App\Models\User;
use App\Models\Company;
use App\Models\ScheduleSlot;
use Carbon\Carbon;
use Illuminate\Http\Request;

class ScheduleController extends Controller
{
    /* =========================
     * INDEX
     * ========================= */
    public function index()
    {
        $authUser = auth()->user();

        $query = Schedule::with(['user', 'company'])
            ->where('status', true);

        if ($authUser->hasRole('Super Admin')) {
            // See all schedules
        }
        elseif ($authUser->can('schedules.view.all')) {
            $query->where('company_id', $authUser->profile->company_id);
        }
        elseif ($authUser->can('schedules.view.own')) {
            $query->where('company_id', $authUser->profile->company_id)
                  ->where('user_id', $authUser->id);
        }
        else {
            abort(403);
        }

        $schedules = $query
            ->orderBy('schedule_date', 'desc')
            ->paginate(15);

        return view('schedules.index', compact('schedules'));
    }

    /* =========================
     * CALENDAR
     * ========================= */
    public function calendar()
    {
        $authUser = auth()->user();

        $query = Schedule::with('user');

        if ($authUser->hasRole('Super Admin')) {
            // no filter
        }
        elseif ($authUser->can('schedules.view.all')) {
            $query->where('company_id', $authUser->profile->company_id);
        }
        elseif ($authUser->can('schedules.view.own')) {
            $query->where('company_id', $authUser->profile->company_id)
                  ->where('user_id', $authUser->id);
        }
        else {
            abort(403);
        }

        $schedules = $query->get();

        return view('schedules.calendar', compact('schedules'));
    }

    /* =========================
     * CREATE
     * ========================= */
    public function create()
    {
        $authUser = auth()->user();

        if ($authUser->hasRole('Super Admin')) {
            $users = User::orderBy('name')->get();
            $companies = Company::orderBy('name')->get();
        } else {
            $companyId = $authUser->profile->company_id;

            $users = User::whereHas('profile', function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            })->orderBy('name')->get();

            $companies = Company::where('id', $companyId)->get();
        }

        return view('schedules.create', compact('users', 'companies'));
    }

    /* =========================
     * STORE
     * ========================= */
    public function store(Request $request)
    {
        $authUser = auth()->user();

        $data = $request->validate([
            'schedule_date' => 'required|date',
            'morning_location' => 'required|string|max:255',
            'morning_start_time' => 'required|date_format:H:i',
            'morning_end_time' => 'required|date_format:H:i|after:morning_start_time',

            'evening_location' => 'nullable|string|max:255',
            'evening_start_time' => 'nullable|date_format:H:i',
            'evening_end_time' => 'nullable|date_format:H:i|after:evening_start_time',

            'per_customer_time' => 'required|integer|min:1',
            'fees' => 'required|numeric|min:0',

            'user_id' => 'required|exists:users,id',
            'company_id' => 'required|exists:companies,id',
            'status' => 'required|boolean',
        ]);

        /* 🔒 Enforce security */
        if (!$authUser->hasRole('Super Admin')) {
            $data['company_id'] = $authUser->profile->company_id;

            if (!$authUser->can('schedules.change.user')) {
                $data['user_id'] = $authUser->id;
            }
        }

        // Prevent duplicate schedule for same user/day
        $exists = Schedule::where('user_id', $data['user_id'])
            ->whereDate('schedule_date', $data['schedule_date'])
            ->exists();

        if ($exists) {
            return back()->withInput()
                ->withErrors(['user_id' => 'This user already has a schedule on this date.']);
        }

        $schedule = Schedule::create($data);

        $this->generateSlotsForSchedule($schedule, $data);

        return redirect()
            ->route('admin.schedules.index')
            ->with('success', 'Schedule created successfully.');
    }

    /* =========================
     * EDIT
     * ========================= */
    public function edit(Schedule $schedule)
    {
        $authUser = auth()->user();

        if (!$authUser->hasRole('Super Admin')) {
            abort_if(
                $schedule->company_id !== $authUser->profile->company_id,
                403
            );
        }

        if ($authUser->hasRole('Super Admin')) {
            $users = User::orderBy('name')->get();
            $companies = Company::orderBy('name')->get();
        } else {
            $companyId = $authUser->profile->company_id;

            $users = User::whereHas('profile', function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            })->orderBy('name')->get();

            $companies = Company::where('id', $companyId)->get();
        }

        return view('schedules.edit', compact('schedule', 'users', 'companies'));
    }

    /* =========================
     * UPDATE
     * ========================= */
    public function update(Request $request, Schedule $schedule)
    {
        $authUser = auth()->user();

        $data = $request->validate([
            'schedule_date' => 'required|date',
            'per_customer_time' => 'required|integer|min:1',
            'fees' => 'required|numeric|min:0',
            'status' => 'required|boolean',

            'morning_location' => 'required|string',
            'morning_start_time' => 'required',
            'morning_end_time' => 'required',

            'evening_location' => 'nullable|string',
            'evening_start_time' => 'nullable',
            'evening_end_time' => 'nullable',

            'user_id' => 'required|exists:users,id',
            'company_id' => 'required|exists:companies,id',
        ]);

        if (!$authUser->hasRole('Super Admin')) {
            $data['company_id'] = $authUser->profile->company_id;

            if (!$authUser->can('schedules.change.user')) {
                $data['user_id'] = $schedule->user_id;
            }
        }

        $schedule->update($data);

        return redirect()
            ->route('admin.schedules.index')
            ->with('success', 'Schedule updated successfully.');
    }

    /* =========================
     * DELETE
     * ========================= */
    public function destroy(Schedule $schedule)
    {
        $schedule->slots()->delete();
        $schedule->delete();

        return redirect()
            ->route('admin.schedules.index')
            ->with('success', 'Schedule deleted successfully.');
    }

    /* =========================
     * SHOW
     * ========================= */
    public function show(Schedule $schedule)
    {
        $slots = $schedule->slots()->orderBy('start_time')->get();
        return view('schedules.show', compact('schedule', 'slots'));
    }

    /* =========================
     * SLOT GENERATION
     * ========================= */
    protected function generateSlotsForSchedule(Schedule $schedule, array $data)
    {
        $this->generateSlots(
            $schedule,
            'morning',
            $data['morning_location'],
            $data['morning_start_time'],
            $data['morning_end_time'],
            $data['per_customer_time'],
            $data['fees']
        );

        if (!empty($data['evening_start_time'])) {
            $this->generateSlots(
                $schedule,
                'evening',
                $data['evening_location'],
                $data['evening_start_time'],
                $data['evening_end_time'],
                $data['per_customer_time'],
                $data['fees']
            );
        }
    }

    protected function generateSlots(
        Schedule $schedule,
        string $shift,
        string $location,
        string $startTime,
        string $endTime,
        int $perCustomerTime,
        float $fees
    ) {
        $date = $schedule->schedule_date->format('Y-m-d');

        $start = Carbon::parse("$date $startTime");
        $end = Carbon::parse("$date $endTime");

        while ($start->lt($end)) {
            $slotEnd = $start->copy()->addMinutes($perCustomerTime);
            if ($slotEnd->gt($end)) {
                $slotEnd = $end->copy();
            }

            ScheduleSlot::create([
                'schedule_id' => $schedule->id,
                'user_id' => $schedule->user_id,
                'company_id' => $schedule->company_id,
                'shift' => $shift,
                'location' => $location,
                'start_time' => $start->format('H:i:s'),
                'end_time' => $slotEnd->format('H:i:s'),
                'fees' => $fees,
            ]);

            $start->addMinutes($perCustomerTime);
        }
    }
}
