<?php

namespace App\Http\Controllers;

use App\Models\Service;
use App\Models\Category;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\View\View;

class ServiceController extends Controller
{
    /**
     * Display a listing of services.
     */
    public function index(): View
    {
        $services = Service::with(['category', 'company', 'parent'])
            ->orderBy('order')
            ->paginate(20);

        return view('services.index', compact('services'));
    }

    /**
     * Show the form for creating a new service.
     */
    public function create(): View
    {
        return view('services.create', [
            'categories' => Category::where('status', 1)->get(),
            'companies'  => Company::where('status', 1)->get(),
            'parents'    => Service::whereNull('parent_id')->get(),
        ]);
    }

    /**
     * Store a newly created service.
     */
    public function store(Request $request): RedirectResponse
    {
        $data = $this->validatedData($request);

        $data['slug'] = Str::slug($data['name']);
        $data['user_id'] = auth()->id();

        $this->handleUploads($request, $data);

        Service::create($data);

        return redirect()
            ->route('admin.services.index')
            ->with('success', 'Service created successfully.');
    }

    /**
     * Show the form for editing the specified service.
     */
    public function edit(Service $service): View
    {
        return view('services.edit', [
            'service'    => $service,
            'categories' => Category::where('status', 1)->get(),
            'companies'  => Company::where('status', 1)->get(),
            'parents'    => Service::whereNull('parent_id')
                ->where('id', '!=', $service->id)
                ->get(),
        ]);
    }

    /**
     * Update the specified service.
     */
    public function update(Request $request, Service $service): RedirectResponse
    {
        $data = $this->validatedData($request, $service->id);

        $data['slug'] = Str::slug($data['name']);

        $this->handleUploads($request, $data, $service);

        $service->update($data);

        return redirect()
            ->route('admin.services.index')
            ->with('success', 'Service updated successfully.');
    }

    /**
     * Remove the specified service.
     */
    public function destroy(Service $service): RedirectResponse
    {
        $this->deleteFiles($service);

        $service->delete();

        return back()->with('success', 'Service deleted successfully.');
    }

    /**
     * Validate request data.
     */
    protected function validatedData(Request $request, int $id = null): array
    {
        return $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|unique:services,slug,' . $id,

            'category_id' => 'nullable|exists:categories,id',
            'parent_id'   => 'nullable|exists:services,id',
            'company_id'  => 'nullable|exists:companies,id',

            'status' => 'boolean',
            'order'  => 'nullable|integer',

            'icon'   => 'nullable|image|mimes:png,jpg,jpeg,webp,svg|max:2048',
            'image'  => 'nullable|image|mimes:png,jpg,jpeg,webp|max:4096',
            'banner' => 'nullable|image|mimes:png,jpg,jpeg,webp|max:4096',

            'short_description' => 'nullable|string|max:255',
            'description'       => 'nullable|string',

            'meta' => 'nullable|array',
            'data' => 'nullable|array',
        ]);
    }

    /**
     * Handle file uploads.
     */
    protected function handleUploads(
        Request $request,
        array &$data,
        ?Service $service = null
    ): void {
        foreach (['icon', 'image', 'banner'] as $field) {
            if ($request->hasFile($field)) {

                // Delete old file
                if ($service && $service->{$field}) {
                    Storage::disk('public')->delete($service->{$field});
                }

                $data[$field] = $request
                    ->file($field)
                    ->store("services/{$field}s", 'public');
            }
        }
    }

    /**
     * Delete all service files.
     */
    protected function deleteFiles(Service $service): void
    {
        foreach (['icon', 'image', 'banner'] as $field) {
            if ($service->{$field}) {
                Storage::disk('public')->delete($service->{$field});
            }
        }
    }
}
