<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Company;
use App\Models\Profile;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    /**
     * Display a listing of users.
     */
    public function index()
    {
        $users = User::with(['roles', 'profile.company', 'profile.reportsTo'])->latest()->get();
        return view('users.index', compact('users'));
    }

    /**
     * Show the form for creating a new user.
     */
    public function create()
    {
        $roles = Role::pluck('name', 'name');
        $companies = Company::orderBy('name')->get();
        $users = User::orderBy('name')->get(); // for "Reports To"

        return view('users.create', compact('roles', 'companies', 'users'));
    }

    /**
     * Store a newly created user.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name'               => 'required|string|max:255',
            'email'              => 'required|email|unique:users,email',
            'password'           => 'required|min:6|confirmed',
            'roles'              => 'required|array',
            'company_id'         => 'nullable|exists:companies,id',
            'reports_to_user_id' => 'nullable|exists:users,id',
            'phone'              => 'nullable|string|max:20',
            'reference'          => 'nullable|string|max:255',
            'position'           => 'nullable|string|max:255',
            'department'         => 'nullable|string|max:255',
            'about'              => 'nullable|string',
        ]);

        /** 1️⃣ Create user */
        $user = User::create([
            'name'     => $request->name,
            'email'    => $request->email,
            'password' => Hash::make($request->password),
        ]);

        /** 2️⃣ Assign roles */
        $user->syncRoles($request->roles);

        /** 3️⃣ Create profile */
        Profile::updateOrCreate(
            ['user_id' => $user->id],
            [
                'company_id'         => $request->company_id,
                'reports_to_user_id' => $request->reports_to_user_id,
                'phone'              => $request->phone,
                'reference'          => $request->reference,
                'position'           => $request->position,
                'department'         => $request->department,
                'about'              => $request->about,
            ]
        );

        return redirect()
            ->route('admin.users.index')
            ->with('success', 'User created successfully');
    }

    /**
     * Show the form for editing the user.
     */
    public function edit(User $user)
    {
        $roles = Role::pluck('name', 'name');
        $companies = Company::orderBy('name')->get();
        $users = User::orderBy('name')->get(); // for "Reports To"
        $userRoles = $user->roles->pluck('name')->toArray();
        $profile = $user->profile ?? new Profile();

        return view('users.edit', compact('user', 'roles', 'userRoles', 'companies', 'users', 'profile'));
    }

    /**
     * Update the user.
     */
    public function update(Request $request, User $user)
    {
        $request->validate([
            'name'               => 'required|string|max:255',
            'email'              => 'required|email|unique:users,email,' . $user->id,
            'password'           => 'nullable|min:6|confirmed',
            'roles'              => 'required|array',
            'company_id'         => 'nullable|exists:companies,id',
            'reports_to_user_id' => 'nullable|exists:users,id',
            'phone'              => 'nullable|string|max:20',
            'reference'          => 'nullable|string|max:255',
            'position'           => 'nullable|string|max:255',
            'department'         => 'nullable|string|max:255',
            'about'              => 'nullable|string',
        ]);

        /** 1️⃣ Update user fields */
        $user->update($request->only('name', 'email'));

        if ($request->filled('password')) {
            $user->update([
                'password' => Hash::make($request->password),
            ]);
        }

        /** 2️⃣ Sync roles */
        $user->syncRoles($request->roles);

        /** 3️⃣ Update or create profile */
        Profile::updateOrCreate(
            ['user_id' => $user->id],
            [
                'company_id'         => $request->company_id,
                'reports_to_user_id' => $request->reports_to_user_id,
                'phone'              => $request->phone,
                'reference'          => $request->reference,
                'position'           => $request->position,
                'department'         => $request->department,
                'about'              => $request->about,
            ]
        );

        return redirect()
            ->route('admin.users.index')
            ->with('success', 'User updated successfully');
    }

    /**
     * Delete user.
     */
    public function destroy(User $user)
    {
        $user->delete();

        return redirect()
            ->route('admin.users.index')
            ->with('success', 'User deleted successfully');
    }
}
