@extends('layouts.app')

@section('title', 'Leads')

@section('content')
<div class="container">
    <div class="page-inner">
        <div class="row row-demo-grid">
            <div class="col-md-8">
                <div class="page-header">
                    <h3 class="fw-bold mb-3">Leads</h3>
                    <ul class="breadcrumbs mb-3">
                        <li class="nav-home">
                            <a href="{{ route('dashboard') }}"><i class="icon-home"></i></a>
                        </li>
                        <li class="separator"><i class="icon-arrow-right"></i></li>
                        <li class="nav-item"><a href="#">Leads</a></li>
                    </ul>
                </div>
            </div>

            <div class="col-md-4 ms-auto">
                @can('leads.create')
                    <a href="{{ route('admin.leads.create') }}" class="btn btn-sm btn-primary mb-3">
                        Add Lead
                    </a>
                @endcan
            </div>
        </div>

        {{-- Bulk Actions --}}
        <div class="mb-3 d-flex align-items-center flex-wrap gap-2">
            <input type="checkbox" id="selectAll" class="me-2"> Select All

            @can('leads.change_stage')
            <select id="bulkStageSelect" class="form-select w-auto me-2">
                <option value="">-- Change Stage --</option>
                @foreach($stages as $stg)
                    <option value="{{ $stg->id }}">{{ $stg->name }}</option>
                @endforeach
            </select>
            <button id="applyBulkStage" class="btn btn-sm btn-primary me-2">Apply Stage</button>
            @endcan

            @can('leads.change_user')
            <select id="bulkUserSelect" class="form-select w-auto me-2">
                <option value="">-- Change User --</option>
                @foreach($users as $usr)
                    <option value="{{ $usr->id }}">{{ $usr->name }}</option>
                @endforeach
            </select>
            <button id="applyBulkUser" class="btn btn-sm btn-info me-2">Apply User</button>
            @endcan

            @can('leads.delete')
            <button id="bulkDelete" class="btn btn-sm btn-danger">Delete Selected</button>
            @endcan
        </div>

        {{-- Leads Table --}}
        <div class="card">
            <div class="card-body">
                <div class="table-responsive">
                    <table id="leadsTable" class="display table table-striped table-hover">
                        <thead>
                            <tr>
                                <th></th>
                                <th>Name</th>
                                <th>Phone</th>
                                <th>Stage</th>
                                <th>Status</th>
                                <th>Assigned To</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tfoot>
                            <tr>
                                <th></th>
                                <th>Name</th>
                                <th>Phone</th>
                                <th>Stage</th>
                                <th>Status</th>
                                <th>User</th>
                                <th></th>
                            </tr>
                        </tfoot>
                        <tbody>
                            @forelse($leads as $lead)
                                <tr>
                                    <td><input type="checkbox" class="select-lead" value="{{ $lead->id }}"></td>
                                    <td>{{ $lead->name }}</td>
                                    <td>{{ $lead->phone }}</td>
                                    <td>
                                        <span class="badge bg-info">{{ $lead->stage->name ?? '-' }}</span>
                                    </td>
                                    <td>
                                        <span class="badge bg-secondary">{{ ucfirst($lead->status) }}</span>
                                    </td>
                                    <td>{{ $lead->user->name ?? '-' }}</td>
                                    <td>
                                        {{-- Edit --}}
                                        @can('leads.edit')
                                            <a href="{{ route('admin.leads.edit', $lead) }}" class="btn btn-sm btn-warning">Edit</a>
                                        @endcan

                                        {{-- View --}}
                                        @canany(['leads.view.all', 'leads.view.company', 'leads.view.own'])
                                            <a href="{{ route('admin.leads.show', $lead) }}" class="btn btn-sm btn-info">View</a>
                                        @endcanany

                                        {{-- Delete --}}
                                        @can('leads.delete')
                                            <form action="{{ route('admin.leads.destroy', $lead) }}" method="POST" style="display:inline-block">
                                                @csrf
                                                @method('DELETE')
                                                <button class="btn btn-sm btn-danger" onclick="return confirm('Delete this lead?')">Delete</button>
                                            </form>
                                        @endcan
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="7" class="text-center">No leads found</td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>

                {{ $leads->links() }}
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
$(function () {
    // Initialize DataTable
    var table = $('#leadsTable').DataTable({
        pageLength: 10,
        initComplete: function () {
            this.api().columns([1,2,3,4]).every(function () {
                var column = this;
                var select = $('<select class="form-select form-select-sm"><option value="">All</option></select>')
                    .appendTo($(column.footer()).empty())
                    .on('change', function () {
                        column.search($(this).val()).draw();
                    });

                column.data().unique().sort().each(function (d) {
                    d = d.replace(/<[^>]*>/g, '');
                    if (d) select.append('<option value="'+d+'">'+d+'</option>');
                });
            });
        }
    });

    // Select all checkboxes
    $('#selectAll').on('click', function() {
        $('.select-lead').prop('checked', this.checked);
    });

    // Bulk Stage update
    @can('leads.change_stage')
    $('#applyBulkStage').on('click', function() {
        var selectedLeads = $('.select-lead:checked').map(function(){ return $(this).val(); }).get();
        var newStageId = $('#bulkStageSelect').val();

        if(selectedLeads.length === 0){ alert('Select at least one lead.'); return; }
        if(!newStageId){ alert('Select a stage.'); return; }

        $.post("{{ route('admin.leads.bulk-stage-update') }}", {
            _token: "{{ csrf_token() }}",
            lead_ids: selectedLeads,
            new_stage_id: newStageId
        }, function(){ location.reload(); });
    });
    @endcan

    // Bulk User update
    @can('leads.change_user')
    $('#applyBulkUser').on('click', function() {
        var selectedLeads = $('.select-lead:checked').map(function(){ return $(this).val(); }).get();
        var newUserId = $('#bulkUserSelect').val();

        if(selectedLeads.length === 0){ alert('Select at least one lead.'); return; }
        if(!newUserId){ alert('Select a user.'); return; }

        $.post("{{ route('admin.leads.bulk-user-update') }}", {
            _token: "{{ csrf_token() }}",
            lead_ids: selectedLeads,
            new_user_id: newUserId
        }, function(){ location.reload(); });
    });
    @endcan

    // Bulk Delete
    @can('leads.delete')
    $('#bulkDelete').on('click', function() {
        var selectedLeads = $('.select-lead:checked').map(function(){ return $(this).val(); }).get();
        if(selectedLeads.length === 0){ alert('Select at least one lead.'); return; }
        if(!confirm('Are you sure you want to delete selected leads?')) return;

        $.ajax({
            url: "{{ route('admin.leads.bulk-delete') }}",
            type: 'POST',
            data: {
                _token: "{{ csrf_token() }}",
                lead_ids: selectedLeads
            },
            success: function(){ location.reload(); }
        });
    });
    @endcan
});
</script>
@endsection
